/**
 * @file uart.h
 * @date 2015-12-17
 *
 * NOTE:
 * This file is generated by DAVE. Any manual modification done to this file will be lost when the code is regenerated.
 *
 * @cond
 ***********************************************************************************************************************
 * UART v4.1.10 - Configures a USIC channel to perform transmit & receive operations using UART protocol.
 *
 * Copyright (c) 2015-2016, Infineon Technologies AG
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,are permitted provided that the
 * following conditions are met:
 *
 *   Redistributions of source code must retain the above copyright notice, this list of conditions and the  following
 *   disclaimer.
 *
 *   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
 *   disclaimer in the documentation and/or other materials provided with the distribution.
 *
 *   Neither the name of the copyright holders nor the names of its contributors may be used to endorse or promote
 *   products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE  FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY,OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT  OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * To improve the quality of the software, users are encouraged to share modifications, enhancements or bug fixes
 * with Infineon Technologies AG (dave@infineon.com).
 ***********************************************************************************************************************
 *
 * Change History
 * --------------
 *
 * 2015-02-16:
 *     - Initial version for DAVEv4
 *
 * 2015-06-20:
 *     - Changed the abort API name from UART_Abort_Receive to UART_AbortReceive and added return type
 *
 * 2015-07-01:
 *     - Changed structure name from UART_DYNAMIC_t to UART_RUNTIME_t
 *
 * 2015-07-09:
 *     - Renamed UART_GetProtocolStatus as UART_GetFlagStatus and UART_ClearProtocolStatus as UART_ClearFlag.
 *     - Removed APIs UART_GetTXFIFOStatus, UART_GetRXFIFOStatus, UART_ClearTXFIFOStatus,
 *       UART_ClearRXFIFOStatus
 *
 * 2015-07-14:
 *     - Added support for DMA and Direct mode
 *
 * @endcond
 *
 */

/***********************************************************************************************************************
 * HEADER FILES
 **********************************************************************************************************************/
#ifndef UART_H
#define UART_H

#include <xmc_scu.h>
#include <xmc_gpio.h>
#include <xmc_uart.h>

#if (!((XMC_LIB_MAJOR_VERSION == 2U) && \
       (XMC_LIB_MINOR_VERSION >= 1U) && \
       (XMC_LIB_PATCH_VERSION >= 10U)))
#error "UART requires XMC Peripheral Library v2.1.10 or higher"
#endif

#include <DAVE_common.h>
#include "uart_conf.h"
#if ((defined UART_TX_DMA_USED) || (defined UART_RX_DMA_USED))
#include "../GLOBAL_DMA/global_dma.h"
#endif

/**********************************************************************************************************************
 * MACROS
 **********************************************************************************************************************/
/*
 * @brief Represents the maximum data size for DMA transaction*/
#define UART_DMA_MAXCOUNT (4095U)

/**********************************************************************************************************************
 * ENUMS
 **********************************************************************************************************************/
/**
 * @ingroup UART_enumerations
 * @{
 */

/**
 * @brief Enum to describe the possible status values, returned
 *  by UART APIs.
 */
typedef enum UART_STATUS
{
  UART_STATUS_SUCCESS,        /**< Indicates App initialization state successful */

  UART_STATUS_FAILURE,        /**< Unknown error */

  UART_STATUS_BUSY,           /**< UART Busy */

  UART_STATUS_BUFFER_INVALID, /**< Buffer provided or the buffer size is invalid*/

  UART_STATUS_MODE_MISMATCH   /**< API invoked by a handle configured with different mode.
                                   e.g, If UART_StartTransmitDMA is invoked for an instance
                                   which has transmit mode configured as "Interrupt", will
                                   return this status.*/

} UART_STATUS_t;

/**
 * @brief Enum used to describe the UART Mode of operation
 */
typedef enum UART_MODE
{
  UART_MODE_FULLDUPLEX, /**< Full Duplex mode selected */
  UART_MODE_HALFDUPLEX, /**< Half Duplex mode selected */
  UART_MODE_LOOPBACK    /**< LoopBack mode selected */
} UART_MODE_t;

/**
 * @brief Enum used to identify UART protocol event callback function
 */
typedef enum UART_EVENT
{
  UART_EVENT_SYNC_BRK,    /**< Synchronization break detected event */
  UART_EVENT_RX_NOISE,    /**< Receiver noise detected event */
  UART_EVENT_FORMAT_ERR0, /**< Frame format error at stop bit 0 event */
  UART_EVENT_FORMAT_ERR1, /**< Frame format error at stop bit 1 event */
  UART_EVENT_COLLISION,   /**< Data collision detected in half duplex mode event */
  UART_EVENT_MAX          /**< Indicates number of UART events supported*/
} UART_EVENT_t;

/**
 * @brief Enum used to identify the transfer type used for either transmit or receive function.
 */
typedef enum UART_TRANSFER_MODE
{
  UART_TRANSFER_MODE_INTERRUPT,  /**< Implement data transmit or receive using interrupts */
  UART_TRANSFER_MODE_DMA,        /**< Implement data transmit or receive using DMA */
  UART_TRANSFER_MODE_DIRECT      /**< This configuration exposes signals for external APP connection */
}UART_TRANSFER_MODE_t;

/**
 * @}
 */

/**********************************************************************************************************************
 * DATA STRUCTURES
 **********************************************************************************************************************/
/**
 * @brief Function pointer used for callback function
 */
typedef void (*UART_cbhandler)(void);
/**
 * @brief Function pointer used for initialization function
 */
typedef UART_STATUS_t (*UART_init_handler)(void);
/**
 * @ingroup UART_datastructures
 * @{
 */

/**
 * @brief Wrapper typedefinition for XMC_UART_PROTOCOL_STATUS_t.
 */
typedef XMC_UART_CH_STATUS_FLAG_t UART_PROTOCOL_STATUS_t;

/**
 * @brief Structure for transmit pin configuration.
 */
typedef struct UART_TX_CONFIG
{
  XMC_GPIO_PORT_t *const port;            /**< Pointer to the GPIO port base address */
  const uint8_t pin;                      /**< Pin number in the port*/
  const XMC_GPIO_CONFIG_t *const config;  /**< Pin configuration structure */
} UART_TX_CONFIG_t;

#if (defined(UART_TX_DMA_USED) || defined(UART_RX_DMA_USED))
/**
 * @brief Structure for DMA configuration.
 */
typedef struct UART_DMA_CONFIG
{
  const XMC_DMA_CH_CONFIG_t * dma_ch_config;   /**< Pointer to the DMA channel configuration.*/
  uint8_t dma_channel;                         /**< DMA channel number */
}UART_DMA_CONFIG_t;
#endif

/**
 * @brief Structure for holding the configuration parameters of UART channel.
 */
typedef struct UART_CONFIG
{
  const XMC_UART_CH_CONFIG_t * const channel_config;   /**< Basic UART configuration from the GUI with baud,data bits,
                                                            frame length, stop bits and parity */
#if (defined UART_TX_DMA_USED) || (defined UART_RX_DMA_USED)
  GLOBAL_DMA_t * global_dma;                           /**< Global DMA handle */
#endif
#ifdef UART_TX_DMA_USED
  const UART_DMA_CONFIG_t * const transmit_dma_config; /**< Pointer to the DMA channel configuration used for data transmission.*/
#endif
#ifdef UART_RX_DMA_USED
  const UART_DMA_CONFIG_t * const receive_dma_config;  /**< Pointer to the DMA channel configuration used for data reception.*/
#endif
  UART_init_handler fptr_uart_config;                  /**< Function pointer to configure the MUX values */
#ifdef UART_TX_INTERRUPT_USED
  UART_cbhandler tx_cbhandler;                         /**< Function pointer to hold the callback function pointer,
                                                            called when the transmission is complete */
#endif
#ifdef UART_RX_INTERRUPT_USED
  UART_cbhandler rx_cbhandler;                         /**< Function pointer to hold the callback function pointer,
                                                            called when the reception is complete */
#endif
  UART_cbhandler sync_error_cbhandler;                 /**< Function pointer to hold the callback function pointer,
                                                            called when synchronization break detected.*/
  UART_cbhandler rx_noise_error_cbhandler;             /**< Function pointer to hold the callback function pointer,
                                                            called when receiver noise is detected*/
  UART_cbhandler format_error_bit0_cbhandler;          /**< Function pointer to hold the callback function pointer,
                                                            called when format error with stop bit 0 is detected.*/
  UART_cbhandler format_error_bit1_cbhandler;          /**< Function pointer to hold the callback function pointer,
                                                            called when format error with stop bit 1 is detected.*/
  UART_cbhandler collision_error_cbhandler;            /**< Function pointer to hold the callback function pointer,
                                                            called when collision error is detected*/
  const UART_TX_CONFIG_t * tx_pin_config;              /**< Transmit pin configuration to be used during initialization
                                                            and while changing baudrate. */
  UART_MODE_t mode;                                    /**< UART operation mode */
  UART_TRANSFER_MODE_t transmit_mode;                  /**< Mode used for transmitting data. Data can be transmitted using
                                                            interrupt, DMA or direct(using polling or external APP connection.)*/
  UART_TRANSFER_MODE_t receive_mode;                   /**< Mode used for receiving data. Data can be received using
                                                            interrupt, DMA or direct(using polling or external APP connection.)*/
  XMC_USIC_CH_FIFO_SIZE_t tx_fifo_size;                /**< Transmit FIFO size configuration */
  XMC_USIC_CH_FIFO_SIZE_t rx_fifo_size;                /**< Receive FIFO size configuration */
  uint8_t tx_sr;                                       /**< Service request number assigned to transmit interrupt */
} UART_CONFIG_t;

/**
 * @brief Structure to hold the dynamic variables for the UART communication.
 */
typedef struct UART_RUNTIME
{
  uint8_t * tx_data;            /**< Pointer to the transmit data buffer*/
  uint8_t * rx_data;            /**< Pointer to the receive data buffer*/
  uint32_t tx_data_count;       /**< Number of bytes of data to be transmitted*/
  uint32_t tx_data_index;       /**< Index to the byte to be transmitted next in the tx_data buffer*/
  uint32_t rx_data_count;       /**< Number of bytes of data to be received*/
  uint32_t rx_data_index;       /**< Indicates the number of bytes currently available in the rx_data buffer*/
  volatile bool tx_busy;        /**< Status flag to indicate busy when a transmission is assigned*/
  volatile bool rx_busy;        /**< Status flag to indicate busy when a reception is assigned*/
} UART_RUNTIME_t;

/**
 * @brief Handler structure with pointers to dynamic and static parameters.
 */
typedef struct UART
{
  XMC_USIC_CH_t * const channel;              /**< USIC channel*/
  const UART_CONFIG_t * const config;         /**< UART configuration structure pointer*/
  UART_RUNTIME_t * const runtime;             /**< Pointer to the structure holding all variables,
                                                   that can change at runtime*/
} UART_t;

/**
 * @}
 */

/***********************************************************************************************************************
 * API Prototypes
 **********************************************************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif
/**
 * @ingroup UART_apidoc
 * @{
 */
/**
 * @brief Get the UART APP version.
 * @return DAVE_APP_VERSION_t APP version information (major, minor and patch number)
 *
 * Example Usage:
 * @code
 *  //Description:
 *  //Transmits the text "UART APP supported.", if the UART APP version is v4.1.x, where x can be any value.
 *   #include <DAVE.h>
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     DAVE_APP_VERSION_t uart_version;
 *     uint8_t valid_str[] = "UART APP supported.";
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       uart_version = UART_GetAppVersion();
 *       if((uart_version.major == 4) &&
 *          (uart_version.minor == 1))
 *       {
 *         UART_Transmit(&UART_0, valid_str, sizeof(valid_str));
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 *  @endcode
 */
DAVE_APP_VERSION_t UART_GetAppVersion(void);

/**
 * @brief Initializes the UART module as per the configuration made
 * in UI.
 * @param handle Pointer to static and dynamic content of APP configuration.
 * @return UART_STATUS_t: Status of UART driver initialization.\n
 *                    @ref UART_STATUS_SUCCESS - on successful initialization.\n
 *                    @ref UART_STATUS_FAILURE - if initialization fails.\n
 *                    @ref UART_STATUS_BUSY - if UART channel is busy.\n
 *
 * \par<b>Description:</b><br>
 * Initializes IO pins used for the UART communication, configures
 * USIC registers based on the settings provided in the GUI. Calculates divider
 * values PDIV and STEP for a precise baudrate. It also enables configured interrupt
 * flags and service request values.
 * <BR>
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *         while(1U)
 *         {
 *         }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 *  @endcode
 */
UART_STATUS_t UART_Init(const UART_t *const handle);

/**
 * @brief Registers a request for receiving data over UART channel.
 *
 *
 * @param  handle  Pointer to UART_t handle structure
 * @param data_ptr  Pointer to data of type uint8_t.
 * @param  count  Total no of bytes to be received.\n
 *                \b Range: minimum= 1, maximum= maximum value supported by type uint32_t.
 *
 * @return  UART_STATUS_t: Status for receive request.\n
 *                        @ref UART_STATUS_SUCCESS if the request is accepted.\n
 *                        @ref UART_STATUS_BUSY if a reception is in progress.\n
 *                        @ref UART_STATUS_BUFFER_INVALID if the data_ptr is NULL or count is 0. \n
 *
 *
 * \par<b>Description:</b><br>
 * Data will be received asynchronously. After the requested number of data bytes are received,
 * optionally, the user configured callback function will be executed.
 * Data reception is accomplished using the receive mode selected in the UI.
 * <b>Interrupt:</b><br>
 * Based on the UI configuration, either standard receive buffer(RBUF) or receive FIFO(OUT) is used
 * for data reception. An interrupt is configured for reading received data from the bus. This function
 * only registers a request to receive a number of data bytes from a USIC channel. If FIFO is
 * configured for reception, the FIFO limit is dynamically configured to optimally utilize the
 * CPU load. Before starting data reception, the receive buffers are flushed. So only those data, received
 * after calling the API, will be placed in the user buffer.
 * When all the requested number of data bytes are received, the configured callback
 * function will be executed.
 * If a callback function is not configured, the user has to poll for the value of the
 * variable, \a handle->runtime->rx_busy to be false. The value is updated to \a false when all the
 * requested number of data bytes are received.
 * <br>
 * <b>DMA:</b><br>
 * DMA mode is available only in XMC4x family of microcontrollers. In this mode, a DMA channel is
 * configured for receiving data from standard receive buffer(RBUF) to the user buffer. By calling
 * this API, the DMA channel destination address is configured to the user buffer and the channel is
 * enabled. FIFO will not be used when the receive mode is DMA.
 * Before starting data reception, the receive buffers are flushed. So only those data, received
 * after calling the API, will be placed in the user buffer.
 * When all the requested number of data bytes are received, the configured callback
 * function will be executed.
 * If a callback function is not configured, the user has to poll for the value of the
 * variable, \a handle->runtime->rx_busy to be false. The value is updated to \a false when all the
 * requested number of data bytes are received.
 * <br>
 * <b>Direct</b><br>
 * In Direct receive mode, neither interrupt nor DMA is used. The API polls
 * the receive flag to read the received data and waits for all the requested number of bytes to
 * be received. Based on FIFO configuration, either RBUF or OUT register is used for reading received
 * data. Before starting data reception, the receive buffers are flushed. So only those data, received
 * after calling the API, will be placed in the user buffer.
 * <i><b> Note:</b> In Direct mode, the API blocks the CPU until the count of bytes requested is received.
 * If this behaviour is not desired, use other APIs like @ref UART_GetReceivedWord, @ref UART_GetProtocolStatus etc.</i>
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Receive mode should be configured as "Direct".
 *   //Description:
 *   //Transmits 10 bytes of data after receiving 10 bytes of data.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t ReadData[10];
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       while(1)
 *       {
 *         //Receive 10 bytes of data
 *         if(UART_Receive(&UART_0, ReadData, 10) == UART_STATUS_SUCCESS)
 *         {
 *           //Retransmit the received 10 bytes
 *           UART_Transmit(&UART_0, ReadData, 10);
 *         }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 *  @endcode
 *
 */
UART_STATUS_t UART_Receive(const UART_t *const handle, uint8_t* data_ptr, uint32_t count);

/**
 * @brief Registers a request for transmitting data over UART channel.
 *
 * @param  handle  UART APP handle pointer of type @ref UART_t
 * @param  data_ptr Pointer to data of type uint8_t.
 * @param  count Total no of words to be transmitted.\n
 * \b Range: minimum= 1, maximum= maximum supported by uint32_t.
 *
 * @return  UART_STATUS_t: Status of transmit request.\n
 *                        @ref UART_STATUS_SUCCESS if the request is accepted.\n
 *                        @ref UART_STATUS_BUSY if a transmission is in progress.\n
 *                        @ref UART_STATUS_BUFFER_INVALID if the data_ptr is NULL or count is 0.\n
 *
 * <i>Imp Note:</i> Return value should be validated by user to ensure that the
 * request is registered.
 *
 * \par<b>Description:</b><br>
 * Transmits data using the UART channel. Transmission is accomplished using the transmit mode
 * as configured in the UI.<br>
 * <b>Interrupt:</b><br>
 * The data transmission is accomplished using transmit interrupt. User can configure
 * a callback function in the APP UI. When the data is fully transmitted, the callback
 * function will be executed. If transmit FIFO is enabled, the trigger limit is set to 1.
 * So the transmit interrupt will be generated when all the data in FIFO is moved out of FIFO.
 * The APP handle's runtime structure is used to store the data pointer, count, data index
 * and status of transmission. This function only registers a data transmission request if
 * there is no active transmission in progress. Actual data transmission happens in the transmit
 * interrupt service routine. A trigger is generated for the transmit interrupt to start loading
 * the data to the transmit buffer. If transmit FIFO is configured, the data is filled into the FIFO.
 * Transmit interrupt will be generated subsequently when the transmit FIFO is empty. At this
 * point of time, if there is some more data to be transmitted, it is loaded to the FIFO again.
 * When FIFO is not enabled, data is transmitted one byte at a time. On transmission of each byte
 * an interrupt is generated and the next byte is transmitted in the interrupt service routine.
 * Callback function is executed when all the data bytes are transmitted.
 * If a callback function is not configured, user has to poll for the value of \a tx_busy flag of
 * the APP handle structure( \a handle->runtime->tx_busy ) to check for
 * the completion of data transmission.<br>
 * <b>DMA:</b><br>
 * A DMA channel is configured to provide data to the UART transmit buffer. This removes the load
 * off the CPU. This API will only configure and enable the DMA channel by specifying the data buffer
 * and count of bytes to transmit. Rest is taken care without the CPU's intervention. User can configure
 * a callback function in the APP UI. When the transmission is complete, the callback function will be executed.
 * FIFO will not be used in DMA mode. Transmit buffer interrupt is configured for triggering the
 * DMA channel. So each byte is transmitted in the background through the DMA channel.
 * If the callback function is not configured, \a handle->runtime->tx_busy flag can be checked to
 * verify if the transmission is complete.
 * <b>Direct:</b><br>
 * Data will be transmitted using polling method. Status flags are used to check if data can be transmitted.
 * <i><b> Note:</b> In Direct mode, the API blocks the CPU until the count of bytes requested is transmitted.
 * If this behaviour is not desired, use other APIs like @ref UART_TransmitWord, @ref UART_GetProtocolStatus etc.</i>
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Transmit mode should be configured as "Direct".
 *   //Description:
 *   //Transmits the string "Infineon".
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon";
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Transmit the string.
 *       UART_Transmit(&UART_0, Send_Data, sizeof(Send_Data)-1);
 *       while(1)
 *       {
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
UART_STATUS_t UART_Transmit(const UART_t *const handle, uint8_t* data_ptr, uint32_t count);

#if (defined UART_TX_INTERRUPT_USED || defined UART_TX_DMA_USED)
/**
 * @brief Aborts the ongoing data transmission.
 * @param handle  UART APP handle pointer of type @ref UART_t
 * @return UART_STATUS_t: Result of transmit abort request.\n
 *                       @ref UART_STATUS_SUCCESS if the UART channel transmission is stopped
 *                       successfully.\n
 *                       @ref UART_STATUS_MODE_MISMATCH if transmit mode is Direct.\n
 *
 * \par<b>Description:</b><br>
 * If there is a transmission in progress, it will be stopped. If transmit FIFO is used,
 * the existing data will be flushed. After the transmission is stopped, user can start
 * a new transmission without delay. Abort request for transmit mode direct cannot be serviced since
 * the direct mode transmit API is a blocking call.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Transmit mode should be configured as "Interrupt".
 *   //Description:
 *   //Initiates the transmission of one string, aborts the transmission immediately and
 *   //starts transmission of another string. The receiver might see traces of first string followed,
 *   //by the complete second string.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     //String1
 *     uint8_t Send_Data[] = "Infineon DAVE application.";
 *     //String2
 *     uint8_t NewData[] = "New data message";
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Initiate transmission of first string.
 *       UART_Transmit(&UART_0, Send_Data, sizeof(Send_Data));
 *		 //When the first string is being transmitted,
 *       if(UART_0.runtime->tx_busy)
 *       {
 *         //Stop the transmission of first string.
 *         if(UART_AbortTransmit(&UART_0) == UART_STATUS_SUCCESS)
 *         {
 *           //Start the transmission of second string
 *           UART_Transmit(&UART_0, NewData, sizeof(NewData));
 *           //Wait till the transmission is finished.
 *           while(UART_0.runtime->tx_busy);
 *         }
 *       }
 *       while(1)
 *       {
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
UART_STATUS_t UART_AbortTransmit(const UART_t *const handle);
#endif

#if (defined UART_RX_INTERRUPT_USED || defined UART_RX_DMA_USED)
/**
 * @brief Stops the active data reception sequence.
 * @param handle UART APP handle pointer of type @ref UART_t
 * @return UART_STATUS_t: Result of reception abort request.\n
 *                       @ref UART_STATUS_SUCCESS if the UART channel reception has
 *                       been stopped.\n
 *                       @ref UART_STATUS_MODE_MISMATCH if receive mode is Direct.\n
 *
 * \par<b>Description:</b><br>
 * If a reception is in progress, it will be stopped. When a reception request
 * is active, user will not be able to place a new receive request till the active
 * reception is complete. This API can stop the progressing reception to make
 * a new receive request. Abort request for receive mode direct cannot be serviced since
 * the direct mode receive API is a blocking call.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Transmit mode and receive mode should be configured as "Interrupt".
 *   //Description:
 *   //Transmits the string "Infineon DAVE application." and starts to receive data.
 *   //After receiving first byte, checks the value of first byte. If the first byte is 0x55,
 *   //aborts the ongoing reception and transmits the received byte.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon DAVE application.";
 *     uint8_t Rec_Data[64];
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Transmit the initial string.
 *       UART_Transmit(&UART_0, Send_Data, sizeof(Send_Data));
 *       //Wait till the string is transmitted.
 *       while(UART_0.runtime->tx_busy);
 *
 *       //Start reception of data
 *       UART_Receive(&UART_0, Rec_Data, 15);
 *       //Wait till atleast one byte is received.
 *       while(UART_0.runtime->rx_data_index == 0);
 *       //If the first byte is 0x55,
 *       if(UART_0.runtime->rx_data[0] == 0x55)
 *       {
 *         //Stop the reception
 *         if(UART_AbortReceive(&UART_0) == UART_STATUS_SUCCESS)
 *         {
 *           //Transmit the received one byte.
 *           UART_Transmit(&UART_0, Rec_Data, 1);
 *         }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
UART_STATUS_t UART_AbortReceive(const UART_t *const handle);
#endif

#ifdef UART_RX_INTERRUPT_USED
/**
 * @brief Registers a request for receiving data over UART channel.
 *
 *
 * @param  handle  Pointer to UART_t handle structure
 * @param data_ptr  Pointer to data of type uint8_t.
 * @param  count  Total no of bytes to be received.\n
 *                \b Range: minimum= 1, maximum= maximum value supported by type uint32_t.
 *
 * @return  UART_STATUS_t: Status for receive request.\n
 *                        @ref UART_STATUS_SUCCESS if the request is accepted.\n
 *                        @ref UART_STATUS_BUSY if a reception is in progress.\n
 *                        @ref UART_STATUS_BUFFER_INVALID if the data_ptr is NULL or count is 0. \n
 *
 *
 * \par<b>Description:</b><br>
 * Data will be received asynchronously. When the requested number of data bytes are received,
 * optionally, the user configured callback function will be executed.
 * Based on the UI configuration, either standard receive buffer or receive FIFO is used
 * for data reception. An interrupt is configured for reading received data from the bus. This function
 * only registers a request to receive a number of data bytes from a USIC channel. If FIFO is
 * configured for reception, the FIFO limit is dynamically configured to optimally utilize the
 * CPU load.  When all the requested number of data bytes are received, the configured callback
 * function will be executed.
 * If a callback function is not configured, the user has to poll for the value of the
 * variable, \a handle->runtime->rx_busy to be false. The value is updated to \a false when all the
 * requested number of data bytes are received.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Transmit mode and receive mode should be configured as "Interrupt".
 *   //Description:
 *   //Receives 10 bytes of data and transmits the same.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t ReadData[10];
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       while(1)
 *       {
 *       //Start reception of 10 bytes. The status will be returned success, if the channel is not busy.
 *       if(UART_StartReceiveIRQ(&UART_0, ReadData, 10) == UART_STATUS_SUCCESS)
 *       {
 *         //Wait till the data is received.
 *         while(UART_0.runtime->rx_busy)
 *         {
 *         }
 *         //Transmit the received data.
 *         UART_Transmit(&UART_0, ReadData, 10);
 *         while(UART_0.runtime->tx_busy)
 *         {
 *         }
 *       }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 *  @endcode
 *
 */
UART_STATUS_t UART_StartReceiveIRQ(const UART_t *const handle, uint8_t* data_ptr, uint32_t count);
#endif

#ifdef UART_TX_INTERRUPT_USED
/**
 * @brief Registers a request for transmitting data over UART channel.
 *
 * @param  handle  UART APP handle pointer of type @ref UART_t
 * @param  data_ptr Pointer to data of type uint8_t.
 * @param  count Total no of words to be transmitted.\n
 * \b Range: minimum= 1, maximum= maximum supported by uint32_t.
 *
 * @return  UART_STATUS_t: Status of transmit request.\n
 *                        @ref UART_STATUS_SUCCESS if the request is accepted.\n
 *                        @ref UART_STATUS_BUSY if a transmission is in progress.\n
 *                        @ref UART_STATUS_BUFFER_INVALID if the data_ptr is NULL or count is 0.\n
 *
 * <i>Imp Note:</i> Return value should be validated by user to ensure that the
 * request is registered.
 *
 * \par<b>Description:</b><br>
 * The data transmission is accomplished using transmit interrupt. User can configure
 * a callback function in the APP UI. When the data is fully transmitted, the callback
 * function will be executed. If transmit FIFO is enabled, the trigger limit is set to 1.
 * So the transmit interrupt will be generated when all the data in FIFO is moved from FIFO.
 * The function uses APP handle's runtime structure to store the data pointer, count, data index
 * and status of transmission. This function only registers a data transmission request if
 * there is no active transmission in progress. Actual data transmission happens in the transmit
 * interrupt service routine. A trigger is generated for the transmit interrupt to start loading
 * the data. If transmit FIFO is configured, the data is filled into the FIFO.
 * Transmit interrupt will be generated next time when the transmit FIFO is empty. At this
 * point of time, if there is some more data to be transmitted, it is loaded to the FIFO again.
 * When FIFO is not enabled, data is transmitted one byte at a time. On transmission of each byte
 * an interrupt is generated and the next byte is transmitted in the interrupt service routine.
 * Callback function is executed when all the data bytes are transmitted.
 * If a callback function is not configured, user has to poll for the value of \a tx_busy flag of
 * the APP handle structure( \a handle->runtime->tx_busy ) to check for
 * the completion of data transmission.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Transmit mode should be configured as "Interrupt".
 *   //Description:
 *   //Transmits the string "Infineon"
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon";
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       if(UART_StartTransmitIRQ(&UART_0, Send_Data, sizeof(Send_Data)) == UART_STATUS_SUCCESS)
 *       {
 *         //Wait till the data transmission is ongoing.
 *         while(UART_0.runtime->tx_busy)
 *         {
 *         }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
UART_STATUS_t UART_StartTransmitIRQ(const UART_t *const handle, uint8_t* data_ptr, uint32_t count);
#endif

#ifdef UART_TX_DMA_USED
/**
 * @brief Registers a request for transmitting data over UART channel using DMA.
 *
 * @param  handle  UART APP handle pointer of type @ref UART_t
 * @param  data_ptr Pointer to data of type uint8_t.
 * @param  count Total number of words to be transmitted.\n
 * \b Range: minimum= 1, maximum= 4096.
 *
 * @return  UART_STATUS_t: Status of transmit request.\n
 *                        @ref UART_STATUS_SUCCESS if the request is accepted.\n
 *                        @ref UART_STATUS_BUSY if a transmission is in progress.\n
 *                        @ref UART_STATUS_MODE_MISMATCH if the handle is not configured for
 *                        DMA transmit mode. \n
 *                        @ref UART_STATUS_BUFFER_INVALID if the data_ptr is NULL or count is 0. \n
 *
 * <i>Imp Note:</i> Return value should be validated by user to ensure that the
 * request is registered.
 *
 * \par<b>Description:</b><br>
 * The data transmission is accomplished using DMA. User can configure
 * a callback function in the APP UI. When the data is fully transmitted, the callback
 * function will be executed.
 * The function uses APP handle's runtime structure to store the status of transmission.
 * This function only registers a data transmission request, if
 * there is no active transmission in progress. Actual data transmission happens through DMA channel.
 * A maximum of 4096 bytes can be transmitted in one API call. This limit is because of the DMA single block size.
 * Callback function is executed when all the data bytes are transmitted.
 * If a callback function is not configured, user has to poll for the value of \a tx_busy flag of
 * the APP handle structure( \a handle->runtime->tx_busy ) to check for
 * the completion of data transmission.
 * If data more than the block size of 4096 have to be transmitted, user will have to transmit them using multiple calls to this API.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Transmit mode should be configured as "DMA".
 *   //Description:
 *   //Transmits the string "Infineon" using DMA.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon";
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       if(UART_StartTransmitDMA(&UART_0, Send_Data, sizeof(Send_Data)) == UART_STATUS_SUCCESS)
 *       {
 *         //Wait till the data is transmitted.
 *         while(UART_0.runtime->tx_busy)
 *         {
 *         }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
UART_STATUS_t UART_StartTransmitDMA(const UART_t *const handle, uint8_t* data_ptr, uint32_t count);
#endif

#ifdef UART_RX_DMA_USED
/**
 * @brief Registers a request for receiving data over UART channel using DMA.
 *
 *
 * @param  handle  Pointer to UART_t handle structure
 * @param data_ptr  Pointer to data of type uint8_t.
 * @param  count  Total no of bytes to be received.\n
 *                \b Range: minimum= 1, maximum= 4096.
 *
 * @return  UART_STATUS_t: Status for receive request.\n
 *                        @ref UART_STATUS_SUCCESS if the request is accepted.\n
 *                        @ref UART_STATUS_BUSY if a reception is in progress.\n
 *                        @ref UART_STATUS_MODE_MISMATCH if the handle is not configured for
 *                        DMA receive mode. \n
 *                        @ref UART_STATUS_BUFFER_INVALID if the data_ptr is NULL or count is 0. \n
 *
 *
 * \par<b>Description:</b><br>
 * Data will be received asynchronously. When the requested number of data bytes are received,
 * optionally, the user configured callback function will be executed.
 * This function only registers a request to receive a number of data bytes from a USIC channel.
 * When all the requested number of data bytes are received, the configured callback
 * function will be executed.
 * If a callback function is not configured, the user has to poll for the value of the
 * variable, \a handle->runtime->rx_busy to be false. The value is updated to \a false when all the
 * requested number of data bytes are received.
 * A maximum of 4096 bytes can be received in one API call. This limit is because of the DMA single block size.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre condition:
 *   //Transmit mode and receive mode should be configured to "DMA"
 *   //Description:
 *   //Receives 10 bytes of data and transmits the received data.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t ReadData[10];
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       while(1)
 *       {
 *         //Receive 10 bytes of data
 *         if(UART_StartReceiveDMA(&UART_0, ReadData, 10) == UART_STATUS_SUCCESS)
 *         {
 *           //Wait till 10 bytes are received
 *           while(UART_0.runtime->rx_busy)
 *           {
 *           }
 *           //Transmit the received data.
 *           UART_StartTransmitDMA(&UART_0, ReadData, 10);
 *         }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 *  @endcode
 *
 */
UART_STATUS_t UART_StartReceiveDMA(const UART_t *const handle, uint8_t* data_ptr, uint32_t count);
#endif

/**
 * @brief Changes the baudrate of UART channel.
 *
 * @param UART_t * Pointer to the UART APP handle.
 * @param baud Value of new baudrate.
 * @param oversampling Number of samples to be considered for each symbol. 16 is the standard value.
 *
 * @return UART_STATUS_t UART_STATUS_SUCCESS if baudrate changed successfully.
 *                       UART_STATUS_FAILURE if baudrate could not be changed.
 *
 * \par<b>Description:</b><br>
 * The function stops the channel, calculates the clock divider values to achieve the desired baudrate.
 * Sets the divider values and reconfigures the channel as per the configuration in the UI. The channel is
 * enabled at the end of configuration.
 * <BR>
 *
 * Example Usage:
 * <i>Please disable the receive FIFO in the 'Advanced Settings' tab</i>
 * @code
 *
 * #include <DAVE.h>      //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 * //Precondition:
 * //Disable receive FIFO in the Advanced settings tab.
 * //Description:
 * //Waits for user input of new baudrate value. Input is recognized after line feed is provided.
 * //Value is set as the new baudrate and the application waits for any key to be pressed.
 * //This helps in reconfiguring the terminal application to the newly set baudrate. On receiving
 * //new character, message indicating the successful baudrate change will be displayed using updated
 * //value of baudrate.
 *
 * const uint8_t send_askbaud[] = "Please enter desired baudrate:";
 * const uint8_t send_data[] = "\nPress 'y' to change baudrate to desired value:";
 * const uint8_t send_invalid[] = "\nInvalid value!!";
 * const uint8_t send_success[] = "\nWe made it...Baudrate changed successfully :-).\n\n";
 * uint8_t rec_data[11];
 * int main(void)
 * {
 *   DAVE_STATUS_t status;
 *   uint32_t baud;
 *
 *   status = DAVE_Init();    // Initialization of DAVE Apps
 *
 *   if(status == DAVE_STATUS_FAILURE)
 *   {
 *     XMC_DEBUG(("DAVE Apps initialization failed with status %d\n", status));
 *     while(1U)
 *     {
 *     }
 *   }
 *
 *   while(1U)
 *   {
 *     UART_Transmit(&UART_0, send_askbaud, sizeof(send_askbaud)-1);
 *     UART_Receive(&UART_0, rec_data, 10);
 *     while(UART_0.runtime->tx_busy);
 *     while(UART_0.runtime->rx_busy)
 *     {
 *       //If user enters newline character, accept the value
 *       if((UART_0.runtime->rx_data_index > 0) && (UART_0.runtime->rx_data[UART_0.runtime->rx_data_index - 1] == 0x0a))
 *       {
 *         //End reception of data on finding newline character
 *         UART_AbortReceive(&UART_0);
 *       }
 *     }
 *     //Add end of string character to the last location
 *     rec_data[UART_0.runtime->rx_data_index] = 0;
 *     //Convert the entered string to number.
 *     baud = atoi(rec_data);
 *     //If the conversion is successful, set the baudrate.
 *     if(baud > 0)
 *     {
 *       //Set the baudrate to USIC channel
 *       if(UART_SetBaudrate(&UART_0, baud, 16) == UART_STATUS_SUCCESS)
 *       {
 *         //After changing the baudrate successfully,
 *         //Wait for user to enter a character.
 *         //This wait gives time for the user to change
 *         //the baudrate of the terminal tool used.
 *         UART_Receive(&UART_0, rec_data, 1);
 *         while(UART_0.runtime->rx_busy);
 *         UART_Transmit(&UART_0, send_success, sizeof(send_success)-1);
 *       }
 *       else
 *       {
 *         UART_Transmit(&UART_0, send_invalid, sizeof(send_invalid)-1);
 *       }
 *     }
 *     else
 *     {
 *       UART_Transmit(&UART_0, send_invalid, sizeof(send_invalid)-1);
 *     }
 *     while(UART_0.runtime->tx_busy);
 *   }
 * }
 * @endcode
 * */
UART_STATUS_t UART_SetBaudrate(const UART_t * handle, uint32_t baud, uint32_t oversampling);

/**
 * @brief Gets the transmit FIFO event flags.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 *
 * @return uint32_t: Status of the STBI and TBERI bits in TRBSR register in
 *                      their bit positions.\n
 *                      \b Range: Use type @ref XMC_USIC_CH_TXFIFO_EVENT_t for the bitmask of events.
 *
 * \par<b>Description:</b><br>
 * Function reads the value of TRBSR register. It masks the standard transmit buffer
 * interrupt flag and transmit buffer error flag before providing the value.
 * User has to mask the bits of interest before checking the status.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *  //Pre-condition:
 *  //Configure transmit mode as "Direct" with transmit FIFO enabled.
 *  //Description:
 *  //Transmits the string "Infineon".
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon";
 *     uint8_t index = 0;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *        //Clear the Tx FIFO standard transmit buffer event.
 *        UART_ClearTXFIFOStatus(&UART_0, XMC_USIC_CH_TXFIFO_EVENT_STANDARD);
 *        //Iterate for the length of the string "Infineon"
 *        while(index < sizeof(Send_Data)-1)
 *        {
 *           //Put the character in the transmit FIFO.
 *           XMC_USIC_CH_TXFIFO_PutData((XMC_USIC_CH_t *)UART_0.channel,(uint16_t)Send_Data[index]);
 *           index++;
 *
 *           //Wait for FIFO transmit standard buffer interrupt to fill it again with remaining data
 *           while((UART_GetTXFIFOStatus(&UART_0) & XMC_USIC_CH_TXFIFO_EVENT_STANDARD) == 0);
 *           UART_ClearTXFIFOStatus(&UART_0, XMC_USIC_CH_TXFIFO_EVENT_STANDARD);
 *        }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
__STATIC_INLINE uint32_t UART_GetTXFIFOStatus(const UART_t* const handle)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return XMC_USIC_CH_TXFIFO_GetEvent(handle->channel);
}

/**
 * @brief Provides the received data from receive buffer.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 * @return uint8_t: Data read from RBUF.\n
 *
 * \par<b>Description:</b><br>
 * This can be used in receive mode "Direct" to read the received data.
 * If Rx FIFO is not configured, function reads the value of RBUF register.
 * Otherwise it reads the data from OUTR register.
 * User can poll for receive event or configure an interrupt by connecting the
 * external INTERRUPT APP to receive event signals. This API can be used inside the ISR to read the received data.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Pre-condition:
 *   //Configure transmit mode and receive mode as "Direct" with transmit FIFO and receive FIFO enabled
 *   //Description:
 *   //Transmits the string "Infineon", receives 10 bytes and retransmits the received 10 bytes.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon";
 *     uint8_t Rec_Data[10];
 *     uint8_t index = 0;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *        //Transmit the string "Infineon"
 *        while(index < sizeof(Send_Data))
 *        {
 *           UART_TransmitWord(&UART_0,Send_Data[index]);
 *           index++;
 *
 *           //Wait for transmit buffer interrupt to fill it again with remaining data
 *           while((UART_GetTXFIFOStatus(&UART_0) & XMC_USIC_CH_TXFIFO_EVENT_STANDARD) == 0);
 *           UART_ClearTXFIFOStatus(&UART_0, XMC_USIC_CH_TXFIFO_EVENT_STANDARD);
 *        }
 *
 *        //Configure receive FIFO trigger limit to 9.
 *        UART_SetRXFIFOTriggerLimit(&UART_0, 9);
 *        //Receive 10 bytes input
 *        index = 0;
 *        //Wait till 10 bytes are received
 *        while(!(UART_GetRXFIFOStatus(&UART_0) &
 *             (XMC_USIC_CH_RXFIFO_EVENT_STANDARD | XMC_USIC_CH_RXFIFO_EVENT_ALTERNATE)))
 *        {
 *          Rec_Data[index] = UART_GetReceivedWord(&UART_0);
 *          index++;
 *          if(index == 10)
 *          {
 *            break;
 *          }
 *
 *        }
 *
 *        //Transmit the received data
 *        index = 0;
 *        while(index < 10)
 *        {
 *           UART_TransmitWord(&UART_0,Rec_Data[index]);
 *           index++;
 *
 *           //Wait for transmit buffer interrupt to fill it again with remaining data
 *           while((UART_GetTXFIFOStatus(&UART_0) & XMC_USIC_CH_TXFIFO_EVENT_STANDARD) == 0);
 *           UART_ClearTXFIFOStatus(&UART_0, XMC_USIC_CH_TXFIFO_EVENT_STANDARD);
 *        }
 *
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
__STATIC_INLINE uint8_t UART_GetReceivedWord(const UART_t* const handle)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return (uint8_t)XMC_UART_CH_GetReceivedData(handle->channel);
}

/**
 * @brief Transmits a word of data.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t \n
 * @param data Data to be transmitted.\n
 * @return None\n
 *
 * \par<b>Description:</b><br>
 * Transmits a byte of data through the UART channel.
 * If Tx FIFO is configured, the data is placed in the IN[0] register of the USIC channel used.
 * If Tx FIFO is not configured, API waits for the TBUF to be free and then places the data in the TBUF register.
 * User can poll for receive event or configure interrupt by connecting an
 * external INTERRUPT APP. This API can be used inside the ISR to read the received data.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Precondition:
 *   //Configure transmit mode and receive mode as "Direct"
 *   //Description:
 *   //Transmits the string "Infinon"
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon";
 *     uint8_t Rec_Data[10];
 *     uint8_t index = 0;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *        while(index < sizeof(Send_Data))
 *        {
 *           UART_TransmitWord(&UART_0,Send_Data[index]);
 *           index++;
 *
 *           //Wait for transmit buffer interrupt to fill it again with remaining data
 *           while((UART_GetTXFIFOStatus(&UART_0) & XMC_USIC_CH_TXFIFO_EVENT_STANDARD) == 0);
 *           UART_ClearTXFIFOStatus(&UART_0, XMC_USIC_CH_TXFIFO_EVENT_STANDARD);
 *        }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
__STATIC_INLINE void UART_TransmitWord(const UART_t* const handle, uint8_t data)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_UART_CH_Transmit(handle->channel, (uint16_t)data);
}

/**
 * @brief Enables the selected protocol events for interrupt generation.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 * @param  events Protocol events to be enabled for interrupt generation. \n
 *         \bRange: Use type \aXMC_UART_CH_EVENT_t to select the event. Multiple events can be
 *         combined using the bitwise OR operation.\n
 * @return None\n
 *
 * \par<b>Description:</b><br>
 * Enables the events by configuring CCR or PCR register based on the event.
 * When the event is enabled, an interrupt can be generated on occurrence of the event.
 * The API should be used only for \a Direct mode related events. Using this API for non \a Direct mode
 * may not yield expected result.
 *
 * Example Usage:
 * @code
 * * #include <DAVE.h>                 //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 * // Precondition:
 * // Disable receive FIFO.
 * // Add INTERRUPT APP and connect the UART APP protocol interrupt signal to the INTERRUPT APP
 * // irq signal. Provide the callback function name in INTERRUPT APP as "ProtocolInterrupt".
 * //
 * // Description:
 * // Generates an event when data loss is detected and transmits a relevent message.
 * // To generate the event, transmit more than 2 bytes of data to the UART channel .
 *
 * int main(void)
 * {
 *   DAVE_STATUS_t status;
 *
 *   status = DAVE_Init();
 *
 *   if(status == DAVE_STATUS_FAILURE)
 *   {
 *     XMC_DEBUG("DAVE APPs initialization failed\n");
 *
 *     while(1U)
 *     {
 *
 *     }
 *   }
 *   //Enable the interrupt generation when data loss is detected.
 *   UART_EnableEvent(&UART_0, XMC_UART_CH_EVENT_DATA_LOST);
 *   while(1U)
 *   {
 *   }
 * }
 * //Protocol interrupt handler
 * void ProtocolInterrupt()
 * {
 *   uint8_t txt_msg[]="Receiver data loss detected";
 *   //Transmit the message to indicate data loss
 *   UART_Transmit(&UART_0, txt_msg, sizeof(txt_msg));
 * }
 * @endcode
 */
__STATIC_INLINE void UART_EnableEvent(const UART_t* const handle, uint32_t events)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_UART_CH_EnableEvent(handle->channel, events);
}

/**
 * @brief Disables selected events from generating interrupt.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 * @param  events Events to be disabled from generating interrupt. \n
 *         \bRange: Use type \aXMC_UART_CH_EVENT_t to select the event. Multiple events can be
 *         combined using the bitwise OR operation.\n
 * @return None
 *
 * \par<b>Description:</b><br>
 * Events are disabled by clearing their respective bits in either CCR, TBCTR or RBCTR.
 *
 * Example Usage:
 * @code
 * #include <DAVE.h>                 //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 * // Precondition:
 * // Add INTERRUPT APP and connect the UART APP protocol interrupt signal to the INTERRUPT APP
 * // irq signal. Provide the callback function name in INTERRUPT APP as "ProtocolInterrupt".
 * //
 * // Description: The example configures protocol interrupt for data loss detection.
 * // When the data loss interrupt occurs, the receive FIFO is cleared. After the receive FIFO
 * // is cleared, the channel can receive few bytes till the FIFO gets filled.
 *
 * int main(void)
 * {
 *   DAVE_STATUS_t status;
 *
 *   status = DAVE_Init();
 *
 *   if(status == DAVE_STATUS_FAILURE)
 *   {
 *     XMC_DEBUG("DAVE APPs initialization failed\n");
 *
 *     while(1U)
 *     {
 *
 *     }
 *   }
 *   //Enable interrupt generation when data loss is detected
 *   UART_EnableEvent(&UART_0, XMC_UART_CH_EVENT_DATA_LOST);
 *   while(1U)
 *   {
 *   }
 * }
 * void ProtocolInterrupt()
 * {
 *   uint8_t txt_msg[]="Receiver data loss detected";
 *   UART_Transmit(&UART_0, txt_msg, sizeof(txt_msg));
 *   UART_DisableEvent(&UART_0, XMC_UART_CH_EVENT_DATA_LOST);
 *   //Clear receive FIFO so that data will be received.
 *   XMC_USIC_CH_RXFIFO_Flush(UART_0.channel);
 *   UART_EnableEvent(&UART_0, XMC_UART_CH_EVENT_DATA_LOST);
 * }
 * @endcode
 *
 */
__STATIC_INLINE void UART_DisableEvent(const UART_t* const handle, uint32_t events)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_UART_CH_DisableEvent(handle->channel, events);
}

/**
 * @brief Checks if the transmit FIFO is full.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 *
 * @return bool Status of transmit FIFO filling level.
 *              \bRange: \atrue- if transmit FIFO is full.<br>
 *                       \afalse- if transmit FIFO is not full.<br>
 * \par<b>Description:</b><br>
 * Checks the status using the register TRBSR. Can be used while filling
 * data to the transmit FIFO.
 *
 * Example Usage:
 * @code
 *  #include <DAVE.h>                 //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *  // Precondition:
 *  // Transmit mode should be "Direct"
 *  //Description:
 *  //Transmits a string using FIFO.
 *
 * uint8_t send_data[] = "Infineon Technologies";
 * int main(void)
 * {
 *   DAVE_STATUS_t status;
 *   uint32_t loc_index;
 *
 *   status = DAVE_Init();           // Initialization of DAVE APPs
 *
 *   if(status == DAVE_STATUS_FAILURE)
 *   {
 *     XMC_DEBUG("DAVE APPs initialization failed\n");
 *
 *     while(1U)
 *     {
 *
 *     }
 *   }
 *   for(loc_index = 0; loc_index < sizeof(send_data); loc_index++)
 *   {
 *     //Wait when Tx FIFO is full
 *     while(UART_IsTXFIFOFull(&UART_0))
 *     {
 *     }
 *     UART_TransmitWord(&UART_0, send_data[loc_index]);
 *   }
 *   while(1U)
 *   {
 *   }
 * }
 * @endcode
 */
__STATIC_INLINE bool UART_IsTXFIFOFull(const UART_t* const handle)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return XMC_USIC_CH_TXFIFO_IsFull(handle->channel);
}

/**
 * @brief Checks if the receive FIFO is empty.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 *
 * @return bool Status of receive FIFO filling level.
 *              \bRange: \atrue- if receive FIFO is empty.<br>
 *                       \afalse- if receive FIFO still has data.<br>
 * \par<b>Description:</b><br>
 * Checks the status using the register TRBSR. Can be used while reading
 * data from the receive FIFO.
 *
 * Example Usage:
 * @code
 *  #include <DAVE.h>                 //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *  // Precondition:
 *  // Receive mode should be "Direct"
 *  //
 *  // Description:
 *  // Receives 10 bytes and transmits the received 10 bytes.
 *
 * uint8_t send_text[] = "Enter 10 bytes:";
 * uint8_t rec_data[10];
 * int main(void)
 * {
 *   DAVE_STATUS_t status;
 *   uint32_t loc_index;
 *
 *   status = DAVE_Init();           // Initialization of DAVE APPs
 *
 *   if(status == DAVE_STATUS_FAILURE)
 *   {
 *     XMC_DEBUG("DAVE APPs initialization failed\n");
 *
 *     while(1U)
 *     {
 *
 *     }
 *   }
 *   UART_Transmit(&UART_0, send_text, sizeof(send_text));
 *   for(loc_index = 0; loc_index < sizeof(rec_data); loc_index++)
 *   {
 *     //Wait when Rx FIFO is empty
 *     while(UART_IsRXFIFOEmpty(&UART_0))
 *     {
 *     }
 *     rec_data[loc_index] = UART_GetReceivedWord(&UART_0);
 *   }
 *   //Transmit the received data
 *   UART_Transmit(&UART_0, rec_data, sizeof(rec_data));
 *   while(1U)
 *   {
 *   }
 * }
 * @endcode
 */
__STATIC_INLINE bool UART_IsRXFIFOEmpty(const UART_t* const handle)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return XMC_USIC_CH_RXFIFO_IsEmpty(handle->channel);
}

/**
 * @brief Configures trigger limit for the transmit FIFO.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 * @param  limit Value of transmit FIFO filling level, transition below which the interrupt should be generated.\n
 *               \bRange: 0 to transmit FIFO size.\n
 *               e.g, If transmit FIFO size is 16, and limit is configured as 8, FIFO standard transmit buffer interrupt
 *               will be generated when the FIFO filling level drops from 8 to 7.\n
 *
 * @return None\n
 *
 * \par<b>Description:</b><br>
 * Transmit FIFO trigger limit is configured by setting its value in the TBCTR register.
 * Transmit FIFO is configured to generate interrupt when the FIFO filling level drops
 * below the trigger limit.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Precondition:
 *   //Configure receive mode as "Direct"
 *   //Add an instance of the INTERRUPT APP and connect the UART event_fifo_transmit_buffer_interrupt
 *   //signal to the INTERRUPT sr_irq signal.
 *   //Provide the callback function name in INTERRUPT APP as "tx_cb"
 *   //Description:
 *   //Transmits the string "Infineon" using FIFO. Configures the FIFO to generate event when the FIFO
 *   //is empty. Puts one byte to the FIFO when the event is generated.
 *
 *   uint8_t Send_Data[] = "Infineon";
 *   uint8_t index = 0;
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Rec_Data[10];
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Configure transmit interrupt generation when the transmit FIFO is empty
 *       UART_SetTXFIFOTriggerLimit(&UART_0, 1);
 *       //Put one word to the FIFO
 *       UART_TransmitWord(&UART_0,Send_Data[index]);
 *       index++;
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 *
 *  //INTERRUPT APP callback function
 *  void tx_cb()
 *  {
 *    if(index < sizeof(Send_Data))
 *    {
 *      //Put one word to the FIFO
 *      UART_TransmitWord(&UART_0,Send_Data[index]);
 *      index++;
 *    }
 *  }
 * @endcode
 */
__STATIC_INLINE void UART_SetTXFIFOTriggerLimit(const UART_t* const handle, uint32_t limit)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_USIC_CH_TXFIFO_SetSizeTriggerLimit(handle->channel, handle->config->tx_fifo_size, limit);
}

/**
 * @brief Configures trigger limit for the receive FIFO.
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 * @param  limit Value of receive FIFO filling level, transition above which the interrupt should be generated.\n
 *               \bRange: 0 to receive FIFO size.\n
 *               e.g, If receive FIFO size is 16, and limit is configured as 8, FIFO receive buffer interrupt
 *               will be generated when the FIFO filling level rises from 8 to 9.\n
 *
 * @return None\n
 *
 * \par<b>Description:</b><br>
 * Receive FIFO trigger limit is configured by setting its value in the RBCTR register.
 * Receive FIFO is configured to generate interrupt when the FIFO filling level rises
 * above the trigger limit.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *
 *   //Precondition:
 *   //Configure receive mode as "Direct"
 *   //Add an instance of the INTERRUPT APP and connect the UART event_fifo_receive_buffer_interrupt
 *   //signal to the INTERRUPT sr_irq signal.
 *   //Provide the callback function name in INTERRUPT APP as "rx_cb"
 *   //Description:
 *
 *   uint8_t Rec_Data[10];
 *   uint8_t index = 0;
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Configure receive interrupt generation when 10 bytes are received
 *       UART_SetRXFIFOTriggerLimit(&UART_0, 9);
 *       //Wait for the data to be received.
 *       while(index < 10);
 *       //Transmit the received data
 *       UART_Transmit(&UART_0, Rec_Data, 10);
 *       //Wait for transmission to finish
 *       while(UART_0.runtime->tx_busy);
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 *
 *  //INTERRUPT APP callback function
 *  void rx_cb()
 *  {
 *    while((index < 10) && (!XMC_USIC_CH_RXFIFO_IsEmpty(UART_0.channel)))
 *    {
 *      //Read data from FIFO
 *      Rec_Data[index] = UART_GetReceivedWord(&UART_0);
 *      index++;
 *    }
 *  }
 * @endcode
 */
__STATIC_INLINE void UART_SetRXFIFOTriggerLimit(const UART_t* const handle, uint32_t limit)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_USIC_CH_RXFIFO_SetSizeTriggerLimit(handle->channel, handle->config->rx_fifo_size, limit);
}

/**
 * @brief Gets the status of event flags related to receive FIFO.
 *
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 *
 * @return uint32_t: Status of standard receive buffer event, alternative receive buffer
 * event and receive buffer error event in their bit positions in TRBSR register.\n
 * \b Range: Use type @ref XMC_USIC_CH_RXFIFO_EVENT_t for event bitmasks. Multiple events' status
 * can be combined for comparison using \a OR operation.
 *
 * \par<b>Description:</b><br>
 * It provides the status of standard receive buffer event, alternative receive buffer
 * event and receive buffer error event.
 * Function masks the TRBSR register with the bitmask of SRBI,
 * ARBI and RBERI flags. User has to mask the bits of interest before checking
 * the status.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *  //Precondition: Configure transmit mode and receive mode as direct.
 *  //Description: Receives data of 10 bytes and retransmits it.
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t ReadData[10];
 *     uint8_t index = 0;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Configure the receive FIFO event to generate when one byte is received.
 *       UART_SetRXFIFOTriggerLimit(&UART_0, 0);
 *       while(1U)
 *       {
 *         //Check if receive FIFO event is generated
 *         if(UART_GetRXFIFOStatus(&UART_0) & XMC_USIC_CH_RXFIFO_EVENT_STANDARD)
 *         {
 *           UART_ClearRXFIFOStatus(&UART_0, XMC_USIC_CH_RXFIFO_EVENT_STANDARD);
 *           //Read received data
 *           ReadData[index] = (uint8_t)XMC_USIC_CH_RXFIFO_GetData((XMC_USIC_CH_t *)&UART_0.channel);
 *           //Transmit received data
 *           UART_Transmit(&UART_0, &ReadData[index], 1);
 *           index++;
 *           index = index % 10;
 *         }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
__STATIC_INLINE uint32_t UART_GetRXFIFOStatus(const UART_t* const handle)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return XMC_USIC_CH_RXFIFO_GetEvent(handle->channel);
}

/**
 * @brief Function clears the specified FIFO event flag related to
 * transmit FIFO.
 *
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 * @param  flag Value with event bits at their bit positions in TRBSR register
 *             to be cleared.\n
 *             \b Range: Use type @ref XMC_USIC_CH_TXFIFO_EVENT_t. Multiple events can be
 *             combined using \a OR operation.
 *
 * @return    None
 *
 * \par<b>Description:</b><br>
 * Function clears a status bit in TRBSR register using the TRBSCR register.
 * But the function does not mask the input value with the bit positions restricted
 * to transmit FIFO status bits. User should ensure that the input value is appropriately
 * masked.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *   //Precondition: Configure transmit mode as direct.
 *   //Description: Transmits the string "Infineon" bytewise.
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t Send_Data[] = "Infineon";
 *     uint8_t index = 0;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *        while(index < sizeof(Send_Data))
 *        {
 *           //Put every byte to FIFO.
 *           XMC_USIC_CH_TXFIFO_PutData(UART_0.channel,(uint16_t)Send_Data[index]);
 *           index++;
 *
 *           //Wait for FIFO transmit standar buffer interrupt to fill it again with remaining data
 *           while((UART_GetTXFIFOStatus(&UART_0) & XMC_USIC_CH_TXFIFO_EVENT_STANDARD) == 0);
 *           UART_ClearTXFIFOStatus(&UART_0, XMC_USIC_CH_TXFIFO_EVENT_STANDARD);
 *        }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
__STATIC_INLINE void UART_ClearTXFIFOStatus(const UART_t* const handle, const uint32_t flag)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_USIC_CH_TXFIFO_ClearEvent(handle->channel, flag);
}

/**
 * @brief Function clears the specified FIFO event flag related to
 * receive FIFO. It should be used to clear the status of standard
 * receive buffer interrupt, alternative receive buffer interupt and
 * receive buffer error interrupt flags.
 *
 *
 * @param  handle UART APP handle pointer of type @ref UART_t
 * @param  flag Value with event bits at the bit positions in TRBSR register
 *             to be cleared.\n
 *             \b Range: Use type @ref XMC_USIC_CH_RXFIFO_EVENT_t for providing events.
 *             Multiple events can be input by using \a OR operation.
 *
 * @return    None
 *
 * \par<b>Description:</b><br>
 * Function clears a status bit in TRBSR using the TRBSCR register.
 * The function does not mask the input value to clear only receive buffer
 * events. So user should appropriately mask the input value before calling
 * the function.
 *
 * Example Usage:
 * @code
 *   #include <DAVE.h>         //Declarations from DAVE Code Generation (includes SFR declaration)
 *   //Precondition: Configure transmit mode and receive mode as direct.
 *   //Description: Receives data of 10 bytes and transmits the same bytewise.
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t ReadData[10];
 *     uint8_t index = 0;
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Configure receive FIFO to generate event when one byte is received.
 *       UART_SetRXFIFOTriggerLimit(&UART_0, 0);
 *       while(1U)
 *       {
 *         //Check if receive event is generated
 *         if(UART_GetRXFIFOStatus(&UART_0) & XMC_USIC_CH_RXFIFO_EVENT_STANDARD)
 *         {
 *           //Clear receive event
 *           UART_ClearRXFIFOStatus(&UART_0, XMC_USIC_CH_RXFIFO_EVENT_STANDARD);
 *           //Read received data from FIFO
 *           ReadData[index] = (uint8_t)XMC_USIC_CH_RXFIFO_GetData((XMC_USIC_CH_t *)&UART_0.channel);
 *           //Transmit received data
 *           UART_Transmit(&UART_0, &ReadData[index], 1);
 *           index++;
 *           index = index % 10;
 *         }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 */
__STATIC_INLINE void UART_ClearRXFIFOStatus(const UART_t* const handle, const uint32_t flag)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_USIC_CH_RXFIFO_ClearEvent(handle->channel, flag);
}

/**
 * @brief Provides the status of protocol events.
 *
 * @param handle UART APP handle pointer of type @ref UART_t
 * @param protocol_status Event whose status is to be read.\n
 *             \b Range: Use type @ref XMC_UART_CH_STATUS_FLAG_t for input. Multiple events can
 *             be combined using \a OR operation.
 *
 * @return uint32_t: Status of selected protocol events read from PSR_ASCMode register.\n
 * \b Range: Use type @ref XMC_UART_CH_STATUS_FLAG_t for comparing the return value
 * with event bitmasks. Status of multiple events can be checked by combining enum
 * values using \a OR operation while comparing.
 *
 * \par<b>Description:</b><br>
 * Reads the protocol status bits from the register PSR_ASCMode and compares the values with
 * the input value of selected events. Returns the masked value of selected events with the status register value.
 * This function is an inline wrapper for the API provided by xmc_uart.h file.
 * <BR>
 *
 * Example Usage:
 * @code
 * #include <DAVE.h>
 *  //Precondition: Configure transmit mode as Interrupt
 *  //Desription: Transmits the string "Infineon", waits for transmit buffer to go idle and then receives 10 bytes.
 *  //Transmits the received 10 bytes.
 *
 * int main(void)
 * {
 *   DAVE_STATUS_t init_status;
 *   uint8_t Send_Data[] = "Infineon";
 *   uint8_t ReceiveData[10] = {0};
 *
 *   init_status = DAVE_Init();
 *
 *   if(init_status == DAVE_STATUS_SUCCESS)
 *   {
 *      //Send the first string.
 *      //Check if the request to transmit is accepted.
 *      while(UART_Transmit(&UART_0,Send_Data, sizeof(Send_Data)) == UART_STATUS_BUSY)
 *      {
 *      }
 *
 *     while(1U)
 *     {
 *        //Check if transmit buffer is idle
 *        if(UART_GetFlagStatus(&UART_0, XMC_UART_CH_STATUS_FLAG_TRANSMISSION_IDLE))
 *        {
 *           //Check if receive request is successful
 *           if(UART_Receive(&UART_0, ReceiveData, 10) == UART_STATUS_SUCCESS)
 *           {
 *             //Wait for reception of 10 bytes
 *             while(UART_0.runtime->rx_busy)
 *             {
 *             }
 *             //Transmit the received data.
 *             UART_Transmit(&UART_0, ReceiveData, 10);
 *           }
 *        }
 *     }
 *   }
 *   else
 *   {
 *
 *     XMC_DEBUG("main: Application initialization failed");
 *     while(1U)
 *     {
 *     }
 *   }
 *
 *   return 1U;
 * }
 * @endcode
 *
 * */
__STATIC_INLINE uint32_t UART_GetFlagStatus(const UART_t * const handle, uint32_t protocol_status)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return (XMC_UART_CH_GetStatusFlag(handle->channel) & protocol_status);
}

/**
 * @brief Clears the event status in the register(PSR_ASCMode).
 *
 * @param handle UART APP handle pointer of type @ref UART_t
 * @param protocol_status Event whose status is to be cleared.\n
 *             \b Range: Use type @ref XMC_UART_CH_STATUS_FLAG_t for input. Multiple events can
 *             be combined using \a OR operation.
 * @return None
 *
 * \par<b>Description:</b><br>
 * Clears a given protocol event flag bit using the PSCR register.
 * This function is an inline wrapper for the API provided by xmc_uart.h file.
 * The user should mask the input value based on the events to be cleared.
 * <BR>
 *
 * Example Usage:
 * @code
 * #include <DAVE.h>
 *  //Precondition:
 *  //Configure receive mode as direct and disable receive FIFO and transmit FIFO.
 *  //Description:
 *  //Transmits each received byte of data.
 * int main(void)
 * {
 *   DAVE_STATUS_t init_status;
 *   uint16_t ReceiveData = 0;
 *
 *   init_status = DAVE_Init();
 *
 *   if(init_status == DAVE_STATUS_SUCCESS)
 *   {
 *     while(1U)
 *     {
 *        //Check if data is received
 *        if(UART_GetFlagStatus(&UART_0, (XMC_UART_CH_STATUS_FLAG_RECEIVE_INDICATION |
 *         XMC_UART_CH_STATUS_FLAG_ALTERNATIVE_RECEIVE_INDICATION)))
 *        {
 *           //Read the received data
 *           ReceiveData = XMC_UART_CH_GetReceivedData(UART_0.channel);
 *           //Transmit the received data
 *           XMC_UART_CH_Transmit(UART_0.channel,(const uint16_t)ReceiveData);
 *           //Clear the receive flags
 *           UART_ClearFlag(&UART_0,(XMC_UART_CH_STATUS_FLAG_RECEIVE_INDICATION |
 *                    XMC_UART_CH_STATUS_FLAG_ALTERNATIVE_RECEIVE_INDICATION));
 *        }
 *     }
 *   }
 *   else
 *   {
 *
 *     XMC_DEBUG("main: Application initialization failed");
 *     while(1U)
 *     {
 *     }
 *   }
 *
 *   return 1U;
 * }
 * @endcode
 *
 * */
__STATIC_INLINE void UART_ClearFlag(const UART_t *const handle, const uint32_t protocol_status)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  XMC_UART_CH_ClearStatusFlag(handle->channel, protocol_status);
}

/**
 * @brief Checks if the transmission is in progress.
 *
 * @param handle UART APP handle pointer of type @ref UART_t
 * @return bool: Status of data transmission.\n
 *               <b>Range:</b> true - if transmission is ongoing.\n
 *                             false- if transmission is not active.\n
 *
 * \par<b>Description:</b><br>
 * Indicates if the communication channel is busy in transmitting data provided using UART_Transmit, UART_StartTransmitIRQ or
 * UART_StartTransmitDMA API.
 * <BR>
 *
 * Example Usage:
 * @code
 * #include <DAVE.h>
 *   //Pre-condition:
 *   //Transmit mode should be configured as "Interrupt".
 *   //Description:
 *   //Initiates the transmission of one string, aborts the transmission immediately and
 *   //starts transmission of another string. The receiver might see traces of first string followed,
 *   //by the complete second string.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     //String1
 *     uint8_t Send_Data[] = "Infineon DAVE application.";
 *     //String2
 *     uint8_t NewData[] = "New data message";
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       //Initiate transmission of first string.
 *       UART_Transmit(&UART_0, Send_Data, sizeof(Send_Data));
 *     //When the first string is being transmitted,
 *       if(UART_IsTxBusy(&UART_0))
 *       {
 *         //Stop the transmission of first string.
 *         if(UART_AbortTransmit(&UART_0) == UART_STATUS_SUCCESS)
 *         {
 *           //Start the transmission of second string
 *           UART_Transmit(&UART_0, NewData, sizeof(NewData));
 *           //Wait till the transmission is finished.
 *           while(UART_IsTxBusy(&UART_0));
 *         }
 *       }
 *       while(1)
 *       {
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 * */
__STATIC_INLINE bool UART_IsTxBusy(const UART_t *const handle)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return (handle->runtime->tx_busy);
}

/**
 * @brief Checks if data reception is in progress.
 *
 * @param handle UART APP handle pointer of type @ref UART_t
 * @return bool: Status of data reception.\n
 *               <b>Range:</b> true - if reception is ongoing.\n
 *                             false- if reception is not active.\n
 *
 * \par<b>Description:</b><br>
 * Indicates if the communication channel is configured for receiving data, initiated using UART_Receive, UART_StartReceiveIRQ or
 * UART_StartReceiveDMA API.
 * <BR>
 *
 * Example Usage:
 * @code
 * #include <DAVE.h>
 *
 *   //Pre-condition:
 *   //Transmit mode and receive mode should be configured as "Interrupt".
 *   //Description:
 *   //Receives 10 bytes of data and transmits the same.
 *
 *   int main(void)
 *   {
 *     UART_STATUS_t init_status;
 *     uint8_t ReadData[10];
 *
 *     init_status = (UART_STATUS_t)UART_Init(&UART_0);
 *     if(init_status == UART_STATUS_SUCCESS)
 *     {
 *       while(1)
 *       {
 *       //Start reception of 10 bytes. The status will be returned success, if the channel is not busy.
 *       if(UART_StartReceiveIRQ(&UART_0, ReadData, 10) == UART_STATUS_SUCCESS)
 *       {
 *         //Wait till the data is received.
 *         while(UART_IsRxBusy(&UART_0))
 *         {
 *         }
 *         //Transmit the received data.
 *         UART_Transmit(&UART_0, ReadData, 10);
 *         while(UART_IsTxBusy(&UART_0))
 *         {
 *         }
 *       }
 *       }
 *     }
 *     else
 *     {
 *      XMC_DEBUG("main: Application initialization failed");
 *      while(1U)
 *      {
 *      }
 *     }
 *      return 1U;
 *  }
 * @endcode
 *
 * */
__STATIC_INLINE bool UART_IsRxBusy(const UART_t *const handle)
{
  XMC_ASSERT("UART APP handle invalid", (handle != NULL))
  return (handle->runtime->rx_busy);
}

/**
 *@}
 */

#ifdef __cplusplus
}
#endif


/* Include App extern declaration file */
#include "uart_extern.h"

#endif /* UART_H_ */
